<?php

namespace App\Http\Controllers\Backend;

use PDF;
use App\Models\User;
use App\Models\Project;
use App\Models\Customer;
use App\Models\Attendance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Artisan;
use RealRashid\SweetAlert\Facades\Alert;

class DashboardController extends Controller
{
    /**
     * @param Request $request
     */
    public function dashboard(Request $request)
    {
        $data['page_title'] = __('Dashboard');
        $data['attend'] = Attendance::whereDate('attend_at', today())->whereUserId(Auth::id())->first();
        $data['totalEmployee'] = User::role('Employee')->count();
        $data['totalProject'] = Project::count();
        $data['customers'] = Customer::all();
        $data['employees'] = User::role('Employee')->get();

        /* filter option */
        if ($request->all()) {
            $data['subTitle'] = '';
            $startDate = null;
            $endDate = null;

            if ($request->filled('daterange')) {
                $daterange = explode('--', $request->input('daterange'));
                $startDate = trim($daterange[0]);
                $endDate = trim($daterange[1]);
            }

            $data['startDate'] = $startDate;
            $data['endDate'] = $endDate;

            $customer = $request->input('customer');
            $employee = $request->input('employee');

            if ($customer) {
                $data['scustomer'] = Customer::find($customer);
            } else {
                $data['scustomer'] = collect();
            }
            if ($employee) {
                $data['semployee'] = User::find($employee);
            } else {
                $data['semployee'] = collect();
            }
            $projects = Project::with(['customer', 'employee'])
                ->when($customer, function ($q) use ($customer) {
                    $q->where('customer_id', $customer);
                })
                ->when($employee, function ($q) use ($employee) {
                    $q->where('employee_id', $employee);
                })
                ->when($request->filled('status'), function ($q) use ($request) {
                    $q->where('status', $request->status);
                })
                ->when($request->daterange, function ($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->orderBy('status')
                ->orderByDesc('id')
                ->paginate()
                ->withQueryString();
        } else {
            $data['subTitle'] = __('Last 20 Project');
            $projects = Project::with(['customer', 'employee'])
                ->when(Auth::user()->hasRole('Employee'), function ($q) {
                    $q->where('employee_id', Auth::id());
                })
                ->orderBy('status')
                ->orderByDesc('id')
                ->paginate();
        }

        $data['projects'] = $projects;
        return view('backend.dashboard.index', $data);
    }

    public function profile()
    {
        $data['page_title'] = "Profile";
        $data['user'] = Auth::user();
        return view('backend.settings.profile', $data);
    }

    /**
     * @param Request $request
     */
    public function language(Request $request)
    {
        $request->validate([
            'lang' => 'required',
        ]);
        App::setLocale($request->lang);
        session()->put('locale', $request->lang);
        $this->updateEnv(['APP_LOCAL' => $request->lang]);

        Alert::success('Well Done.!', 'Language Updated Successfully.');
        return redirect()->back();
    }

    /**
     * @param Request $request
     */
    public function attendance(Request $request)
    {
        $request->validate([
            'attend' => 'required',
        ]);

        if ($request->input('attend') == 1) {
            $attend = 'enabled';
        } else {
            $attend = 'disabled';
        }
        $this->updateEnv(['ATTENDANCE' => $attend]);

        Alert::success('Well Done.!', 'Attendance Updated Successfully.');
        return redirect()->back();
    }

    /**
     * @param array $data
     */
    public function updateEnv($data = [])
    {
        if (count($data) > 0) {
            $path = app()->environmentFilePath();
            $env = file_get_contents($path);
            $env = preg_split('/(\r\n|\r|\n)/', $env);

            foreach ((array) $data as $key => $value) {
                if (preg_match('/\s/', $value) || strpos($value, '#') !== false) {
                    $value = '"' . $value . '"';
                }
                foreach ($env as $env_key => $env_value) {
                    $entry = explode("=", $env_value, 2);
                    if ($entry[0] == $key) {
                        $env[$env_key] = $key . "=" . $value;
                    } else {
                        $env[$env_key] = $env_value;
                    }
                }
            }
            $env = implode("\n", $env);
            file_put_contents($path, $env);

            Artisan::call('config:clear');
            Artisan::call('config:cache');
        }
    }

    /**
     * @param Request $request
     */
    public function report(Request $request)
    {
        $projects = collect([]);
        if ($request->all()) {
            $startDate = null;
            $endDate = null;
            if ($request->filled('daterange')) {
                $daterange = explode('--', $request->input('daterange'));
                $startDate = trim($daterange[0]);
                $endDate = trim($daterange[1]);
            }

            $data['startDate'] = $startDate;
            $data['endDate'] = $endDate;

            $customer = $request->input('customer');
            $employee = $request->input('employee');

            if ($customer) {
                $data['scustomer'] = Customer::find($customer);
            } else {
                $data['scustomer'] = collect();
            }
            if ($employee) {
                $data['semployee'] = User::find($employee);
            } else {
                $data['semployee'] = collect();
            }
            $projects = Project::with(['customer', 'employee'])
                ->when($customer, function ($q) use ($customer) {
                    $q->where('customer_id', $customer);
                })
                ->when($employee, function ($q) use ($employee) {
                    $q->where('employee_id', $employee);
                })
                ->when($request->filled('status'), function ($q) use ($request) {
                    $q->where('status', $request->status);
                })
                ->when($request->daterange, function ($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->orderBy('status')
                ->orderByDesc('id')
                ->paginate()
                ->withQueryString();
        }
        $data['page_title'] = __('Report');
        $data['customers'] = Customer::all();
        $data['employees'] = User::role('employee')->get();
        $data['projects'] = $projects;
        return view('backend.dashboard.report', $data);
    }

    /**
     * @param Request $request
     */
    public function exportPdf(Request $request)
    {
        $request->validate([
            'customer' => 'required',
            'employee' => 'required',
        ]);

        $startDate = $request->input('startDate');
        $endDate = $request->input('endDate');
        $customer = $request->input('customer');
        $employee = $request->input('employee');

        $data['startDate'] = $startDate;
        $data['endDate'] = $endDate;

        if ($customer) {
            $data['scustomer'] = Customer::find($customer);
        } else {
            $data['scustomer'] = collect();
        }
        if ($employee) {
            $data['semployee'] = User::find($employee);
        } else {
            $data['semployee'] = collect();
        }

        $projects = Project::with(['customer', 'employee'])
            ->when($customer, function ($q) use ($customer) {
                $q->where('customer_id', $customer);
            })
            ->when($employee, function ($q) use ($employee) {
                $q->where('employee_id', $employee);
            })
            ->when($request->filled('status'), function ($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($startDate, function ($q) use ($startDate, $endDate) {
                $q->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->orderBy('status')
            ->orderByDesc('id')
            ->get();

        $data['projects'] = $projects;
        // return view('backend.dashboard.report-pdf', $data);
        $pdf = PDF::loadView('backend.dashboard.report-pdf', $data);
        $filename = 'Report ' . time() . '.pdf';
        return $pdf->download($filename);
    }

    /**
     * @param Request $request
     */
    public function deleteReport(Request $request)
    {
        $request->validate([
            'customer' => 'required',
            'employee' => 'required',
        ]);

        $startDate = $request->input('startDate');
        $endDate = $request->input('endDate');
        $customer = $request->input('customer');
        $employee = $request->input('customer');

        Project::with(['customer', 'employee'])
            ->when($employee, function ($q) use ($customer) {
                $q->where('customer_id', $customer);
            })
            ->when($employee, function ($q) use ($employee) {
                $q->where('employee_id', $employee);
            })
            ->when($startDate, function ($q) use ($startDate, $endDate) {
                $q->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->orderByDesc('id')
            ->delete();

        Alert::success(__('Congratulation'), __('Report Delete Successfully.'));
        return to_route('report.index');

    }
}
