<?php

namespace App\Http\Controllers\Backend;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use RealRashid\SweetAlert\Facades\Alert;

class EmployeeController extends Controller
{

    public function __construct()
    {
        $this->middleware('role:Super Admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $data['page_title'] = __('Employee List');
        $data['employees'] = User::role('Employee')
            ->search($request->keyword)
            ->orderBy('id', 'desc')->paginate();
        return view('backend.employee.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $data['page_title'] = __('Create Employee');
        return view('backend.employee.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'     => 'required',
            'email'    => 'required|unique:users,email',
            'password' => 'required|min:6|confirmed',
        ]);

        $in = $request->except(['password_confirmation', 'password']);
        $in['password'] = Hash::make($request->input('password'));

        $user = User::create($in);
        $role = Role::findByName('Employee');
        $user->assignRole($role);

        Alert::success(__('Congratulation'), __('Employee Created Successfully'));
        return to_route('employees.create');

    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $data['page_title'] = __('Edit Employee');
        $data['employee'] = User::findOrFail($id);
        return view('backend.employee.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id): RedirectResponse
    {
        $employee = User::findOrFail($id);
        $request->validate([
            'name'     => 'required',
            'email'    => 'required|unique:users,email,' . $id,
            'password' => 'sometimes|nullable|min:6|confirmed',
        ]);

        $in = $request->except(['password_confirmation', 'password']);

        if ($request->filled('password')) {
            $in['password'] = Hash::make($request->input('password'));
        }
        $employee->update($in);

        Alert::success(__('Congratulation'), __('Employee Updated Successfully'));
        return to_route('employees.index');

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id): RedirectResponse
    {
        // $user = User::findOFail($id);
        User::destroy($id);
        Alert::success(__('Congratulation'), __('Employee Deleted Successfully'));
        return to_route('employees.index');
    }
}
