<?php

namespace App\Http\Controllers\Backend;

use PDF;
use App\Models\User;
use App\Models\Project;
use App\Models\Customer;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\ProjectDiscuss;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Http\RedirectResponse;
use Intervention\Image\Facades\Image;
use RealRashid\SweetAlert\Facades\Alert;
use Illuminate\Contracts\Support\Renderable;

class ProjectController extends Controller
{

    /**
     * @param Request $request
     */
    public function index(Request $request)
    {
        $data['page_title'] = __('List of Project');
        $data['projects'] = Project::with(['customer', 'employee'])
            ->when(Auth::user()->hasRole('Employee'), function ($q) {
                $q->where('employee_id', Auth::id());
            })
            ->search($request->keyword)
            ->orderBy('status')
            ->orderByDesc('id')
            ->paginate()
            ->withQueryString();
        return view('backend.project.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): Renderable
    {
        $data['page_title'] = __('Create New Project');
        $data['employees'] = User::role('Employee')->orderByDesc('id')->get();
        $data['customers'] = Customer::orderByDesc('id')->get();
        return view('backend.project.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $rules = [
            'code'          => 'required',
            'customer_type' => 'required',
            'customer_id'   => 'required_if:customer_type,1',
            'customer_name' => 'required_if:customer_type,2',
            'start_qty'     => 'required|numeric',
            'attachment'    => 'nullable|mimes:jpg,png,jpeg,pdf,doc,docx,xlsx, xls',
            'note'          => 'nullable',
        ];

        if (Auth::user()->hasRole('Super Admin')) {
            $rules = array_merge($rules, [
                'employee_id' => 'required',
            ]);
            $employeeId = $request->input('employee_id');
        } else {
            $employeeId = Auth::id();
        }
        $request->validate($rules);

        $in = $request->except(['customer_type', 'customer_name', 'employee_id']);

        if ($request->input('customer_type') == 2) {
            $customer = Customer::create([
                'name' => $request->input('customer_name'),
            ]);
            $customerId = $customer->id;
        } else {
            $customerId = $request->input('customer_id');
        }

        $in['customer_id'] = $customerId;
        $in['employee_id'] = $employeeId;

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = $request->input('code') . '-Initial-' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('storage/projects'), $filename);
            $in['attachment'] = $filename;
        }
        $in['start_date'] = now();
        Project::create($in);

        Alert::success(__('Congratulation'), __('Project Created Successfully'));
        return to_route('projects.index');

    }

    /**
     * Display the specified resource.
     */
    public function show($id): Renderable
    {
        $data['page_title'] = __('Project Detail');
        $data['project'] = Project::with('discussions.user')->findOrFail($id);
        return view('backend.project.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $project = Project::findOrFail($id);
        $user = Auth::user();
        $readonly = false;
        $isClose = false;

        if ($user->hasRole('Employee')) {
            $readonly = true;
            if ($project->end_date) {
                $isClose = true;
                Alert::warning(__('Oops.!'), __('You can not edit this because of project is Closed.'));
                return redirect()->route('projects.index');
            }
        }

        $data['page_title'] = __('Edit Project');
        $data['project'] = $project;
        $data['readonly'] = $readonly;
        $data['isClose'] = $isClose;
        $data['employees'] = User::role('Employee')->orderByDesc('id')->get();
        $data['customers'] = Customer::orderByDesc('id')->get();
        return view('backend.project.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id): RedirectResponse
    {

        $project = Project::findOrFail($id);

        if (Auth::user()->hasRole('Employee')) {
            $rules = [
                'final_qty' => 'nullable|numeric',
                'note'      => 'nullable',
            ];
        } else {
            $rules = [
                'code'          => 'required',
                'customer_type' => 'required',
                'customer_id'   => 'required_if:customer_type,1',
                'customer_name' => 'required_if:customer_type,2',
                'start_qty'     => 'required|numeric',
                'final_qty'     => 'nullable|numeric',
                'end_date'      => 'nullable|date|date_format:Y-m-d',
                'attachment'    => 'nullable|mimes:jpg,png,jpeg,pdf,doc,docx,xlsx, xls',
                'employee_id'   => 'required',
                'note'          => 'nullable',
            ];
        }

        $request->validate($rules);

        $in = $request->except(['customer_type', 'customer_name', 'customer_phone', 'employee_id']);

        if (Auth::user()->hasRole('Employee')) {
            $project->update([
                'final_qty' => $request->input('final_qty'),
                'note'      => $request->input('note'),
            ]);
            Alert::success(__('Congratulation'), __('Project Updated Successfully'));
            return redirect()->back();
        }

        if ($request->input('customer_type') == 2) {
            $customer = Customer::create([
                'name' => $request->input('customer_name'),
            ]);
            $customerId = $customer->id;
        } else {
            $customerId = $request->input('customer_id');
        }

        $in['customer_id'] = $customerId;
        $in['employee_id'] = $request->input('employee_id');

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = $request->input('code') . '-Initial-' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('storage/projects'), $filename);
            File::delete(public_path("storage/projects/$project->attachment"));
            $in['attachment'] = $filename;
        }

        $project->update($in);

        Alert::success(__('Congratulation'), __('Project Updated Successfully'));
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id): RedirectResponse
    {
        Project::destroy($id);
        Alert::success(trans('Congratulation'), trans('Project Deleted Successfully.'));
        return redirect()->back();
    }

    /**
     * @param $id
     */
    public function download($id)
    {
        $project = Project::findOrFail($id);
        $file = public_path("storage/projects/$project->attachment");
        return response()->download($file);
    }

    /**
     * @param $id
     */
    public function finalDownload($id)
    {
        $project = Project::findOrFail($id);
        $file = public_path("storage/projects/$project->final_attachment");
        return response()->download($file);
    }

    /**
     * @param Request $request
     */
    public function discussions(Request $request)
    {
        $request->validate([
            'project_id' => 'required|exists:projects,id',
            'message'    => 'required',
            'attachment' => 'nullable|mimes:mimes:jpg,png,jpeg,pdf,doc,docx,xlsx, xls',
        ]);

        $project = Project::findOrFail($request->input('project_id'));

        $filename = null;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = Str::random(12) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path("storage/discussions"), $filename);
        }

        $project->discussions()->create([
            'user_id'    => Auth::id(),
            'message'    => $request->input('message'),
            'attachment' => $filename,
        ]);

        Alert::success(__('Congratulation'), __('Project Discussion added Successfully'));
        return redirect()->back();
    }

    /**
     * @param $id
     */
    public function deleteDiscussion($id)
    {
        ProjectDiscuss::destroy($id);
        Alert::success(__('Congratulation'), __('Discussion Deleted Successfully.'));
        return redirect()->back();
    }

    /**
     * @param Request $request
     */
    public function close(Request $request)
    {
        $request->validate([
            'project_id'       => 'required',
            'final_quantity'   => 'required|numeric',
            'final_note'       => 'nullable',
            'final_attachment' => 'required_without:capture_image|mimes:png,jpg,jpeg',
            'capture_image'    => 'required_without:final_attachment',
        ]);

        $project = Project::findOrFail($request->input('project_id'));
        $project->end_date = now();
        $project->status = true;
        $project->final_qty = $request->input('final_quantity');
        $project->final_note = $request->input('final_note');

        if ($request->hasFile('final_attachment')) {
            $file = $request->file('final_attachment');
            $filename = $project->code . '-Final-' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path("storage/projects"), $filename);
            File::delete(public_path("storage/projects/$project->final_attachment"));
        }

        if ($request->filled('capture_image')) {
            $file = $request->capture_image;
            $ext = explode('/', substr($file, 0, strpos($file, ';')))[1];
            $filename = date('Y-m-d H:i:s') . '-' . random_int(1111, 9999) . '.' . $ext;
            Image::make($file)->encode('webp')->save(public_path("storage/projects/{$filename}"));
        }

        $project->final_attachment = $filename;
        $project->save();
        Alert::success(__('Congratulation'), __('Project Close Successfully.'));
        return redirect()->back();
    }

    /**
     * @param $id
     */
    public function discussionDownload($id)
    {
        $dis = ProjectDiscuss::findOrFail($id);
        $file = public_path("storage/discussions/$dis->attachment");
        return response()->download($file);
    }

    /**
     * @param $id
     */
    public function pdf($id)
    {
        $data['project'] = Project::with(['customer', 'employee'])
            ->orderByDesc('id')
            ->findOrFail($id);

        // return view('backend.project.pdf', $data);
        $pdf = PDF::loadView('backend.project.pdf', $data);
        $filename = $data['project']->code . ' ' . time() . '.pdf';
        return $pdf->download($filename);
    }
}
