<?php

namespace App\Models;

use App\Traits\SearchableTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Project extends Model
{
    use HasFactory, SoftDeletes, SearchableTrait;

    /**
     * @var array
     */
    protected $fillable = [
        'title',
        'code',
        'customer_id',
        'start_qty',
        'final_qty',
        'start_date',
        'end_date',
        'employee_id',
        'attachment',
        'note',
        'status',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'start_date' => 'datetime',
        'end_date'   => 'datetime',
    ];

    /**
     * @var array
     */
    protected $searchable = [
        'columns' => [
            'projects.code'  => 5,
            'customers.name' => 5,
            'users.name'     => 5,
            'users.email'    => 5,
        ],
        'joins'   => [
            'customers' => ['customers.id', 'projects.customer_id'],
            'users'     => ['users.id', 'projects.employee_id'],
        ],
    ];

    /**
     * Get the customer that owns the Project
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    /**
     * Get the employee that owns the Project
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function employee(): BelongsTo
    {
        return $this->belongsTo(User::class, 'employee_id');
    }

    /**
     * Get all of the discussions for the Project
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discussions(): HasMany
    {
        return $this->hasMany(ProjectDiscuss::class, 'project_id')->orderBy('id', 'desc');
    }

}
